using System;
using System.Collections.Generic;
using _Core;
using _Core.Storage;
using Newtonsoft.Json;
using Seayoo.ComboSDK.Windows.Account.Model;
using Seayoo.ComboSDK.Windows.Account.Model.VO;

namespace Seayoo.ComboSDK.Windows.Account
{
    public class UserHistoryManager : Singleton<UserHistoryManager>
    {
        private const int MAX_USER_HISTORY_RECORDS = 3;
        private const string userPlayerPrefKey = "combosdk_seayoo_account_users";
        private const string autoLoginKey = "combosdk_seayoo_account_auto_login";

        public void SaveUserHistory(User user)
        {
            var users = GetUserHistories();
            user.time = DateTimeOffset.Now.ToUnixTimeSeconds();
            user.isRecentLogin = true;
            if (users.Count == 0)
            {
                UserPrefs.SetString(
                    userPlayerPrefKey,
                    JsonConvert.SerializeObject(new List<User> { user })
                );
            }
            else
            {
                users.RemoveAll(u => u.userId == user.userId);
                users.Insert(0, user);
                if (users.Count > MAX_USER_HISTORY_RECORDS)
                {
                    users.RemoveAt(users.Count - 1);
                }
                for (int i = 1; i < users.Count; i++)
                {
                    users[i].isRecentLogin = false;
                }
                UserPrefs.SetString(userPlayerPrefKey, JsonConvert.SerializeObject(users));
            }
        }

        public List<User> GetUserHistories()
        {
            if (UserPrefs.HasKey(userPlayerPrefKey))
            {
                var jsonString = UserPrefs.GetString(userPlayerPrefKey);
                Log.I($"User Histories: {jsonString}");
                try
                {
                    var users = JsonConvert.DeserializeObject<List<User>>(jsonString);
                    SetUserHistories(users);
                    return users;
                }
                catch (JsonException ex)
                {
                    Log.E("Error deserializing user data: " + ex.Message);
                }
            }

            return new List<User>();
        }

        public void DeleteUserHistory(string userId)
        {
            var users = GetUserHistories();
            
            if (users.Count == 0)
            {
                UserPrefs.DeleteKey(userPlayerPrefKey);
            }
            else
            {
                users.RemoveAll(u => u.userId == userId);
                UserPrefs.SetString(userPlayerPrefKey, JsonConvert.SerializeObject(users));
            }
        }

        private readonly List<UserHistoryVO> userHistories = new List<UserHistoryVO>();

        public UserHistoryVO FindUserHistory(string mobileMask)
        {
            return userHistories.Find(u => u.mobileMask == mobileMask);
        }

        public UserHistoryVO FindUserHistory(int index)
        {
            return userHistories[index];
        }

        public void SetUserHistories(List<User> users)
        {
            userHistories.Clear();
            foreach (var user in users)
            {
                userHistories.Add(
                    new UserHistoryVO
                    {
                        userId = user.userId,
                        token = user.token,
                        mobileMask = user.mobile,
                        time = user.time,
                        isRecentLogin = user.isRecentLogin
                    }
                );
            }
        }

        public void RemoveUserHistory(string userId)
        {
            userHistories.RemoveAll(user => user.userId == userId);
        }

        public List<UserHistoryVO> GetUsers()
        {
            return userHistories;
        }
    }
}