using System.Collections.Generic;
using ThinkingData.Analytics;
using Combo.Platform;

#if !UNITY_WEBGL
using Newtonsoft.Json;
#endif

namespace Combo.Core
{
    public static class ThinkingDataManager
    {
        [System.Serializable]
        internal class Parameters
        {

#if UNITY_WEBGL
            // WebGL 不使用 Newtonsoft，需要手动映射 key
            public string thinking_data_app_id;
            public string thinking_data_receiver_url;
            public string thinking_data_log_enabled;

            // 兼容旧字段名
            public string appId => thinking_data_app_id;
            public string serverUrl => thinking_data_receiver_url;
            public bool logEnabled =>
                !string.IsNullOrEmpty(thinking_data_log_enabled) &&
                bool.TryParse(thinking_data_log_enabled, out var v1) ? v1 : false;

#else
            [JsonProperty("thinking_data_app_id")]
            public string appId;
            [JsonProperty("thinking_data_receiver_url")]
            public string serverUrl;
            [JsonProperty("thinking_data_log_enabled")]
            public bool logEnabled;
#endif
        }

        internal static void Setup()
        {
            var paramz = Native.GetParameters<Parameters>();

            if (string.IsNullOrEmpty(paramz.appId) || string.IsNullOrEmpty(paramz.serverUrl))
            {
                Log.W($"Initialization of ThinkingDataAnalytics aborted, received empty parameters");
                return;
            }
            TDAnalytics.Init(paramz.appId, paramz.serverUrl);
            TDAnalytics.SetDistinctId(Native.GetDeviceId(), paramz.appId);
            TDAnalytics.EnableLog(paramz.logEnabled);
            SetSuperProperties(
                GetSuperProperties()
            );
            Log.I($"ThinkingDataAnalytics initialized");
        }

        internal static void SetSuperProperties(Dictionary<string, object> properties)
        {
            if(properties == null)
            {
                return;
            }
            Dictionary<string, object> validProperties = new Dictionary<string, object>();

            foreach (var property in properties)
            {
                if (!string.IsNullOrEmpty(property.Value?.ToString()))
                {
                    validProperties[property.Key] = property.Value;
                }
                else
                {
                    TDAnalytics.UnsetSuperProperty(property.Key);
                }
            }

            TDAnalytics.SetSuperProperties(validProperties);
        }

        internal static Dictionary<string,object> GetSuperProperties() {
            var superProperties = new Dictionary<string, object>
            {
                { "distro", Native.GetDistro() },
                { "variant", Native.GetVariant() }
            };

            return superProperties;
        }

        internal static void PutExtraWithDistinctId(Dictionary<string, object> extra)
        {
            extra["thinking_data_distinct_id"] = TDAnalytics.GetDistinctId();
        }
    }
}
