using System;
using _Core;
using _Core.Storage;
using UnityEngine;

namespace _Combo
{
    internal class LanguageManager : Singleton<LanguageManager>
    {
        private LanguagePreference? language;
        private const string languagePrefKey = "combosdk_language_preference";
        public void Setup()
        {
            var value = UserPrefs.GetInt(languagePrefKey);
            language = (LanguagePreference)value;
            Log.I($"local prepence: {Enum.GetName( typeof (LanguagePreference), language)}");
            CheckLanguageMode();
        }

        public void SetLanguagePreference(LanguagePreference language)
        {
            Log.I($"Call SetLanguagePreference(): {Enum.GetName( typeof (LanguagePreference), language)}");
            this.language = language;
            UserPrefs.SetInt(languagePrefKey, (int)language);
            CheckLanguageMode();
        }

        public LanguagePreference GetLanguagePreference()
        {
            return language ?? LanguagePreference.FollowSystem;
        }

        private void CheckLanguageMode()
        {
            switch(GlobalParameters.Instance.SdkLanguageMode)
            {
                case "cn":
                    I18n.SetLanguageCode(LanguageCode.CN);
                    break;
                case "en":
                    I18n.SetLanguageCode(LanguageCode.EN);
                    break;
                case "i18n":
                default:
                    CheckLanguagePreference();
                    break;
            }
        }

        private void CheckLanguagePreference()
        {
            Log.I($"Call CheckLanguagePreference()");
            switch (language)
            {
                case LanguagePreference.ChineseSimplified:
                    I18n.SetLanguageCode(LanguageCode.CN);
                    break;
                case LanguagePreference.English:
                    I18n.SetLanguageCode(LanguageCode.EN);
                    break;
                case LanguagePreference.FollowSystem:
                default:
                    CheckSystemLanguage();
                    break;
            }
        }

        private void CheckSystemLanguage()
        {
            Log.I($"Call CheckSystemLanguage()");
            if(IsSystemLanguageSupported())
            {
                I18n.SetLanguageCode(GetLanguageCode());
            }
            else
            {
                string uiMode = GlobalParameters.Instance.SdkLUIMode;
                LanguageCode code;
                if(uiMode == "china")
                {
                    code = LanguageCode.CN;
                }
                else
                {
                    code = LanguageCode.EN;
                }
                I18n.SetLanguageCode(code);
            }
        }

        private bool IsSystemLanguageSupported()
        {
            if(I18n.IsLanguageSupported(GetISOLanguageCode()))
            {
                return true;
            }
            return false;
        }

        private string GetISOLanguageCode()
        {
            SystemLanguage systemLanguage = Application.systemLanguage;
            switch (systemLanguage)
            {
                case SystemLanguage.ChineseSimplified:
                    return "zh-cn";
                case SystemLanguage.English:
                    return "en";
                default:
                    return "";
            }
        }

        private LanguageCode GetLanguageCode()
        {
            SystemLanguage systemLanguage = Application.systemLanguage;
            switch (systemLanguage)
            {
                case SystemLanguage.ChineseSimplified:
                    return LanguageCode.CN;
                case SystemLanguage.English:
                default:
                    return LanguageCode.EN;
            }
        }
    }
}