using System.Collections;
using UnityEngine;
using UnityEngine.UI;
using _Core;
using Seayoo.ComboSDK.Windows.Account.Model.VO;
using System.Collections.Generic;
using _Combo;

namespace Seayoo.ComboSDK.Windows.Account.View
{
    public class SeayooAccountRealNameView : MonoBehaviour
    {
        public Text titleTxt;
        public HyperlinkText infoTxt;
        public Text nameText;
        public Text idText;
        public Text submitText;
        public InputField nameField;
        public InputField idField;
        public Button submitBtn;
        public Button closeBtn;
        public Image mask;
        public RealNameVO vo;

        void Awake()
        {
            EventSystem.Register(this);
            titleTxt.text = I18n.T("real_name_title");
            nameText.text = I18n.T("real_name_placeholder");
            idText.text = I18n.T("identity_card_placeholder");
            submitText.text = I18n.T("real_name_submit_button");
            gameObject.SetActive(false);
            if(GlobalParameters.Instance.SeayooAccountHideCloseButton)
            {
                closeBtn.gameObject.SetActive(false);
            }
        }

        void OnDestroy() {
            closeBtn.onClick.RemoveListener(OnClose);
            submitBtn.onClick.RemoveListener(OnSubmit);
            nameField.onValueChanged.RemoveListener(OnNameFieldChanged);
            idField.onValueChanged.RemoveListener(OnIdFieldChanged);
            EventSystem.UnRegister(this);
        }

        void Start()
        {
            closeBtn.onClick.AddListener(OnClose);
            submitBtn.onClick.AddListener(OnSubmit);
            nameField.onValueChanged.AddListener(OnNameFieldChanged);
            idField.onValueChanged.AddListener(OnIdFieldChanged);
        }

        [EventSystem.BindEvent]
        void OnGoToView(GoToViewCommand action) {
            if (action.viewType != SeayooAccountViewType.REALNAME) {
                Hide();
                return;
            }
            vo = (RealNameVO)action.vo;
            Show();
        }

        void OnNameFieldChanged(string newValue)
        {
            nameField.text = newValue.Replace(" ", "");
        }

        void OnIdFieldChanged(string newValue)
        {
            idField.text = newValue.Replace(" ", "");
        }

        public void OnSubmit()
        {
            var name = nameField.text;
            var id = idField.text;

            if (string.IsNullOrEmpty(name))
            {
                UIController.Instance.ShowToast(I18n.T("real_name_required_toast"));
                return;
            }
            else if (name.Length < 2)
            {
                UIController.Instance.ShowToast(I18n.T("real_name_invalid_toast"));
                return;
            }
            else if (string.IsNullOrEmpty(id))
            {
                UIController.Instance.ShowToast(I18n.T("identity_card_required_toast"));
                return;
            }
            else if (id.Length < 18)
            {
                UIController.Instance.ShowToast(I18n.T("identity_card_invalid_toast"));
                return;
            }

            submitBtn.interactable = false;
            RealNameCommand.Invoke(new RealNameCommand {
                userId = vo.userId,
                token = vo.token,
                realName = name,
                identityCard = id,
                encrypted = false
            });
        }

        public void OnClose()
        {
            CloseWindowCommand.Invoke(new CloseWindowCommand { userCancel = true });
        }

        [EventSystem.BindEvent]
        public void Refresh(RefreshViewCommand action = null)
        {
            submitBtn.interactable = true;
        }

        public void Show()
        {
            this.gameObject.SetActive(true);
            mask.color = new Color(0f, 0f, 0f, 0.3f);
            infoTxt.text = I18n.T("real_name_text", new Dictionary<string, object>()
            {
                {"0", Config.realNameFurtherNotice},
                {"1", I18n.T("fangchenmi_notification_further_text")}
            });
            StartCoroutine(SetHyperlinkActionsDelayed()); 
        }

        private IEnumerator SetHyperlinkActionsDelayed()
        {
            // 等待一帧，确保 OnPopulateMesh 执行
            yield return null;

            infoTxt.SetHyperlinkClickActions(new Dictionary<string, System.Action<string>>{
                {
                    Config.realNameNotice, (link)=>{
#if UNITY_STANDALONE
                        UIController.Instance.ShowWebView(link, WebViewType.General, 100, 100);
#endif
                }},
                {
                    Config.realNameFurtherNotice, (link)=>{
#if UNITY_STANDALONE
                        UIController.Instance.ShowWebView(link, WebViewType.General, 100, 100);
#endif
                }}
            });
        }

        public void Hide()
        {
            gameObject.SetActive(false);
        }
    }
}
