using System.Text.RegularExpressions;
using UnityEngine;

namespace _Core
{
    // https://learn.microsoft.com/en-us/uwp/api/windows.system.profile.systemmanufacturers.systemsupportdeviceinfo.systemsku?view=winrt-19041
    public class SKU
    {
        public SKU(string manufacturer, string model)
        {
            this.manufacturer = manufacturer;
            this.model = model;
        }
        public string manufacturer;
        public string model;

        public override string ToString()
        {
            return $"{manufacturer}; {model}";
        }
    }

    public class Device
    {
        private static SKU _sku;
        private static string _osVersion;

        public static SKU SKU => _sku ?? (_sku = CreateSKU());

        private static SKU CreateSKU()
        {
            var input = SystemInfo.deviceModel;
            var match = Regex.Match(input, @"^(.*?)\s*\((.*)\)$");
            if (match.Success)
            {
                string model = match.Groups[1].Value.Trim();
                string manufacturer = match.Groups[2].Value.Trim();
                return new SKU(manufacturer, model);
            }
            else
            {
                return new SKU("System manufacturer", "System Product Name");
            }
        }
        public static string OSVersion {
            get {
                var os = SystemInfo.operatingSystem;
                var match = Regex.Match(os, @"(\d+\.\d+\.\d+)");
                return match.Success ? match.Groups[1].Value : os;
            }
        }
        public static string Model => SKU.ToString();
        public static string OSType => SystemInfo.operatingSystemFamily.ToString();
        public static string DeviceName => SystemInfo.deviceName;
        public static string UUID => SystemInfo.deviceUniqueIdentifier;
        public static string LanguageCode => GetLanguageCode(Application.systemLanguage);

        private static string CreateOSVersion()
        {
            var input = SystemInfo.operatingSystem;
            string pattern = @"(?:\()\d+(?:\.\d+)*\.(\d+)(?:\))";
            Match match = Regex.Match(input, pattern);
            if (match.Success)
            {
                // 19045
                return match.Groups[1].Value;
            }
            else
            {
                return input;
            }
        }

        public static string GetLanguageCode(SystemLanguage language)
        {
            switch (language)
            {
                case SystemLanguage.Afrikaans:
                    return "af";
                case SystemLanguage.Arabic:
                    return "ar";
                case SystemLanguage.Basque:
                    return "eu";
                case SystemLanguage.Belarusian:
                    return "be";
                case SystemLanguage.Bulgarian:
                    return "bg";
                case SystemLanguage.Catalan:
                    return "ca";
                case SystemLanguage.Chinese:
                    return "zh";
                case SystemLanguage.ChineseSimplified:
                    return "zh-cn";
                case SystemLanguage.ChineseTraditional:
                    return "zh-tw";
                case SystemLanguage.Czech:
                    return "cs";
                case SystemLanguage.Danish:
                    return "da";
                case SystemLanguage.Dutch:
                    return "nl";
                case SystemLanguage.English:
                    return "en";
                case SystemLanguage.Estonian:
                    return "et";
                case SystemLanguage.Faroese:
                    return "fo";
                case SystemLanguage.Finnish:
                    return "fi";
                case SystemLanguage.French:
                    return "fr";
                case SystemLanguage.German:
                    return "de";
                case SystemLanguage.Greek:
                    return "el";
                case SystemLanguage.Hebrew:
                    return "he";
                case SystemLanguage.Hungarian:
                    return "hu";
                case SystemLanguage.Icelandic:
                    return "is";
                case SystemLanguage.Indonesian:
                    return "id";
                case SystemLanguage.Italian:
                    return "it";
                case SystemLanguage.Japanese:
                    return "ja";
                case SystemLanguage.Korean:
                    return "ko";
                case SystemLanguage.Latvian:
                    return "lv";
                case SystemLanguage.Lithuanian:
                    return "lt";
                case SystemLanguage.Norwegian:
                    return "no";
                case SystemLanguage.Polish:
                    return "pl";
                case SystemLanguage.Portuguese:
                    return "pt";
                case SystemLanguage.Romanian:
                    return "ro";
                case SystemLanguage.Russian:
                    return "ru";
                case SystemLanguage.SerboCroatian:
                    return "sh";
                case SystemLanguage.Slovak:
                    return "sk";
                case SystemLanguage.Slovenian:
                    return "sl";
                case SystemLanguage.Spanish:
                    return "es";
                case SystemLanguage.Swedish:
                    return "sv";
                case SystemLanguage.Thai:
                    return "th";
                case SystemLanguage.Turkish:
                    return "tr";
                case SystemLanguage.Ukrainian:
                    return "uk";
                case SystemLanguage.Vietnamese:
                    return "vi";
                case SystemLanguage.Unknown:
                default:
                    return "unknown";
            }
        }

    }
}

